from __future__ import annotations

import typing as t

from sqlglot import expressions as exp


def dump(expr: exp.Expression) -> t.Dict[str, t.Any]:
    """
    Dump an Expression into a JSON serializable dict.
    """

    root: t.Dict[str, t.Any] = {}
    stack = [(expr, root)]

    while stack:
        node, payload = stack.pop()

        klass = node.__class__.__qualname__

        if node.__class__.__module__ != exp.__name__:
            klass = f"{node.__module__}.{klass}"

        payload["class"] = klass

        if node.type:
            payload["type"] = dump(node.type)
        if node.comments:
            payload["comments"] = node.comments
        if node._meta is not None:
            payload["meta"] = node._meta
        if node.args:
            args: t.Dict[str, t.Any] = {}
            payload["args"] = args

            for k, vs in node.args.items():
                if vs is None or vs == []:
                    continue
                if hasattr(vs, "parent"):
                    args[k] = {}
                    stack.append((vs, args[k]))
                elif type(vs) is list:
                    expressions: t.List[t.Any] = []
                    args[k] = expressions

                    for v in vs:
                        if hasattr(v, "parent"):
                            expressions.append({})
                            stack.append((v, expressions[-1]))
                        else:
                            expressions.append(v)
                elif isinstance(vs, exp.DataType.Type):
                    args[k] = {"class": "DataType.Type", "value": vs.value}
                else:
                    args[k] = vs

    return root


@t.overload
def load(expression: None) -> None: ...


@t.overload
def load(expression: t.Dict[str, t.Any]) -> exp.Expression: ...


def load(expression):
    """
    Load a dict generated by dump into an Expression.
    """
    if expression is None:
        return None

    root = _load(expression)
    stack = [(root, expression.get("args"))]

    while stack:
        obj, args = stack.pop()

        if args:
            for k, vs in args.items():
                if isinstance(vs, list):
                    array = []
                    for v in vs:
                        if isinstance(v, dict):
                            child = _load(v)
                            array.append(child)
                            stack.append((child, v.get("args")))
                        else:
                            array.append(v)
                    obj.set(k, array)
                elif isinstance(vs, dict):
                    child = _load(vs)
                    stack.append((child, vs.get("args")))
                    obj.set(k, child)
                else:
                    obj.set(k, vs)
    return root


def _load(payload: t.Dict[str, t.Any]) -> exp.Expression | exp.DataType.Type:
    class_name = payload["class"]

    if class_name == "DataType.Type":
        return exp.DataType.Type(payload["value"])

    if "." in class_name:
        module_path, class_name = class_name.rsplit(".", maxsplit=1)
        module = __import__(module_path, fromlist=[class_name])
    else:
        module = exp

    expression = getattr(module, class_name)()
    expression.type = load(payload.get("type"))
    expression.comments = payload.get("comments")
    expression._meta = payload.get("meta")
    return expression
