from datetime import datetime
import os
import shutil
from typing import Callable, Optional
import unittest
import warnings

import markdown_it
import pypandoc

from iotas.exporter import Exporter
from iotas.html_generator import HtmlGenerator
from iotas.note import Note
from iotas.pdf_exporter import PdfExporter

warnings.filterwarnings("ignore", "version")


class _MockPdfExporter(PdfExporter):

    note: Note
    location: str

    def set_callbacks(self, finished_callback: Callable, error_callback: Callable) -> None:
        """Set functions to be called upon export result.

        :param Callable finished_callback: Finished callback
        :param Callable error_callback: Error callback
        """
        pass

    def export(self, note: Note, location: str) -> None:
        """Export PDF of note.

        :param Note note: Note to export
        :param str location: Destination location
        """
        self.note = note
        self.location = location


class MockHtmlGenerator(HtmlGenerator):
    """HTML generator mocked."""

    def generate(
        self,
        note: Note,
        searching: bool,
        export_format: Optional[str],
        scroll_position: Optional[float] = None,
    ) -> tuple[str, list[markdown_it.token.Token]]:
        """Generator HTML for note.

        :param Note note: Note to render
        :param bool searching: Whether search CSS should be included
        :param bool export_format: Export format, if using
        :param Optional[float] scroll_position: Position to scroll to
        :return: Generated HTML and list of parser tokens
        :rtype: tuple[str, list[markdown_it.token.Token]]
        """
        return (self.get_output(note), [])

    def generate_user_stylesheet(self, searching: bool) -> str:
        """Generate part of stylesheet based on state (configuration etc).

        :param bool searching: Whether searching
        :return: stylesheet
        :rtype: str
        """
        return ""

    def update_font_family(self, family: str) -> None:
        """Update the font family.

        :param str family: New font family
        """
        pass

    def get_output(self, note: Note) -> str:
        """Get the mocked output.

        :param Note note: The note
        :return: Generated content:
        :rtype: str
        """
        return f"{note.title}|{note.content}"


class Test(unittest.TestCase):

    exporter: Exporter
    pdf_exporter: _MockPdfExporter

    def test_export_md(self) -> None:
        exporter = self.__reset()
        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"

        out_path = self.__prepare_output_dir()

        file_path = os.path.join(out_path, "Test Note Title.md")
        exporter.export(
            note,
            out_format="md",
            file_extension="md",
            supporting_tex=False,
            user_location=file_path,
        )
        self.assertTrue(os.path.exists(file_path))
        with open(file_path, "r") as f:
            contents = f.read()
        self.assertEqual(contents, note.content)

        self.__clean_output_dir()

    def test_export_mock_html(self) -> None:
        exporter = self.__reset()
        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"

        out_path = self.__prepare_output_dir()

        dir_path = os.path.join(out_path, "Test Note Title")
        exporter.export(
            note,
            out_format="html",
            file_extension="html",
            supporting_tex=False,
            user_location=dir_path,
        )
        index_path = os.path.join(dir_path, "index.html")
        self.assertTrue(os.path.exists(index_path))
        with open(index_path, "r") as f:
            contents = f.read()
        self.assertTrue(f"{note.content}" in contents)
        self.assertTrue(f"{note.title}" in contents)

        self.__clean_output_dir()

    def test_mock_export_pdf(self) -> None:
        exporter = self.__reset()
        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"

        out_path = self.__prepare_output_dir()

        file_path = os.path.join(out_path, "Test Note Title.pdf")
        exporter.export(
            note,
            out_format="pdf",
            file_extension="pdf",
            supporting_tex=False,
            user_location=file_path,
        )
        self.assertEqual(self.pdf_exporter.note, note)
        self.assertEqual(self.pdf_exporter.location, file_path)

        self.__clean_output_dir()

    def test_export_odt(self) -> None:
        exporter = self.__reset()
        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"

        out_path = self.__prepare_output_dir()

        file_path = os.path.join(out_path, "Test Note Title.odt")
        exporter.export(
            note,
            out_format="odt",
            file_extension="odt",
            supporting_tex=False,
            user_location=file_path,
        )
        self.assertTrue(os.path.exists(file_path))
        with open(file_path, "rb") as f:
            contents = f.read()

        md = pypandoc.convert_text(contents, to="md", format="odt")
        self.assertTrue(note.content == md.strip())

        self.__clean_output_dir()

    def test_build_default_filename(self) -> None:
        exporter = self.__reset()
        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"

        filename = exporter.build_default_filename(
            note, out_format="md", file_extension="md", add_timestamp=False
        )
        self.assertEqual(filename, "Test Note Title.md")

        filename = exporter.build_default_filename(
            note, out_format="odt", file_extension="odt", add_timestamp=False
        )
        self.assertEqual(filename, "Test Note Title.odt")

        filename = exporter.build_default_filename(
            note, out_format="pdf", file_extension="pdf", add_timestamp=False
        )
        self.assertEqual(filename, "Test Note Title.pdf")

        filename = exporter.build_default_filename(
            note, out_format="html", file_extension="html", add_timestamp=False
        )
        self.assertEqual(filename, "Test Note Title")

        filename = exporter.build_default_filename(
            note, out_format="md", file_extension="md", add_timestamp=True
        )
        parts = filename.split(" ")
        self.assertEqual(len(parts), 4)

        timestamp = None
        try:
            timestamp = datetime.strptime(parts[0], "%Y-%m-%dT%H:%M:%S")
        except ValueError:
            pass

        self.assertIsNotNone(timestamp)
        self.assertEqual(" ".join(parts[1:]), "Test Note Title.md")

    def __reset(self) -> Exporter:
        self.pdf_exporter = _MockPdfExporter()
        html_generator = MockHtmlGenerator()
        data_path = "/mock/data/path"
        self.exporter = Exporter(self.pdf_exporter, html_generator, data_path)
        return self.exporter

    def __prepare_output_dir(self) -> str:
        out_path = self.__get_output_dir()
        if os.path.exists(out_path):
            shutil.rmtree(out_path)
        os.makedirs(out_path)
        return out_path

    def __clean_output_dir(self) -> None:
        out_path = self.__get_output_dir()
        if os.path.exists(out_path):
            shutil.rmtree(out_path)

    def __get_output_dir(self) -> str:
        file_dir = os.path.dirname(__file__)
        return os.path.join(file_dir, os.pardir, "testing-tmp")
