//------------------------------------------------------------------------
//
//  Eureka DOOM Editor
//
//  Copyright (C) 2022 Ioan Chera
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//------------------------------------------------------------------------

#include "im_img.h"
#include "m_game.h"
#include "w_wad.h"
#include "testUtils/Palette.hpp"
#include "gtest/gtest.h"
#include <FL/Fl_RGB_Image.H>

//
// Development helper function to generate the code for the expected image.
// Do NOT make static: we don't want "unused function" errors.
//
void outputPixels(const Img_c &image)
{
	for(int y = 0; y < image.height(); ++y)
	{
		for(int x = 0; x < image.width(); ++x)
		{
			printf("%u, ", image.buf()[y * image.width() + x]);
		}
		puts("");
	}
}

//==================================================================================================

TEST(ImageBasic, PixelMakeRGB)
{
	img_pixel_t pixel = pixelMakeRGB(23, 15, 27);
	ASSERT_EQ(pixel, img_pixel_t(IS_RGB_PIXEL | 23 << 10 | 15 << 5 | 27));

	pixel = pixelMakeRGB(0, 0, 0);
	ASSERT_EQ(pixel, img_pixel_t(IS_RGB_PIXEL));
}

TEST(ImageBasic, ConvertRGBImageNoAlpha)
{
	std::vector<uchar> bits = {
		12, 34, 56,  0,  80,   0, 56, 77,  22,
		98,  0, 98, 22, 255, 200,  0, 99, 255
	};


	Fl_RGB_Image flImage(bits.data(), 3, 2, 3);

	auto image = IM_ConvertRGBImage(flImage);
	ASSERT_FALSE(image->is_null());
	ASSERT_EQ(image->width(), 3);
	ASSERT_EQ(image->height(), 2);
	ASSERT_FALSE(image->has_transparent());
	const img_pixel_t *buf = image->buf();
	ASSERT_EQ(buf[0], pixelMakeRGB(1, 4, 7));
	ASSERT_EQ(buf[1], pixelMakeRGB(0, 10, 0));
	ASSERT_EQ(buf[2], pixelMakeRGB(7, 9, 2));
	ASSERT_EQ(buf[3], pixelMakeRGB(12, 0, 12));
	ASSERT_EQ(buf[4], pixelMakeRGB(2, 31, 25));
	ASSERT_EQ(buf[5], pixelMakeRGB(0, 12, 31));
}

TEST(ImageBasic, ConvertRGBImageWithAlpha)
{
	std::vector<uchar> bits = {
		12, 34, 56, 255,   0,  80,   0, 64,   56, 77,  22, 192,
		98,  0, 98, 200,  22, 255, 200, 255,   0, 99, 255, 32
	};

	Fl_RGB_Image flImage(bits.data(), 3, 2, 4);

	auto image = IM_ConvertRGBImage(flImage);
	ASSERT_FALSE(image->is_null());
	ASSERT_EQ(image->width(), 3);
	ASSERT_EQ(image->height(), 2);
	ASSERT_TRUE(image->has_transparent());
	const img_pixel_t *buf = image->buf();
	ASSERT_EQ(buf[0], pixelMakeRGB(1, 4, 7));
	ASSERT_EQ(buf[1], TRANS_PIXEL);
	ASSERT_EQ(buf[2], pixelMakeRGB(7, 9, 2));
	ASSERT_EQ(buf[3], pixelMakeRGB(12, 0, 12));
	ASSERT_EQ(buf[4], pixelMakeRGB(2, 31, 25));
	ASSERT_EQ(buf[5], TRANS_PIXEL);
}

TEST(ImageBasic, ConvertRGBImageInvalidDepth1)
{
	std::vector<uchar> bits = {
		12, 34, 56,
		98,  0, 98,
	};

	Fl_RGB_Image flImage(bits.data(), 3, 2, 1);
	auto image = IM_ConvertRGBImage(flImage);
	ASSERT_FALSE(image);
}

TEST(ImageBasic, ConvertRGBImageInvalidDepth2)
{
	std::vector<uchar> bits = {
		12, 255, 34, 255, 56, 255,
		98, 255, 0, 0, 98, 255
	};

	Fl_RGB_Image flImage(bits.data(), 3, 2, 2);
	auto image = IM_ConvertRGBImage(flImage);
	ASSERT_FALSE(image);
}

TEST(ImageBasic, ConvertTGAImage)
{
	static const rgba_color_t rgba[] = {
		0xff0080ff, 0x7799aa22, 0x337744ff, 0x12469911,
		0xffffffff, 0x00000000, 0x246732ff, 0x12345678,
	};
	auto image = IM_ConvertTGAImage(rgba, 4, 2);
	ASSERT_FALSE(image.is_null());
	ASSERT_TRUE(image.has_transparent());
	ASSERT_EQ(image.width(), 4);
	ASSERT_EQ(image.height(), 2);
	const auto buf = image.buf();
	ASSERT_NE(buf[0], TRANS_PIXEL);
	ASSERT_EQ(buf[1], TRANS_PIXEL);
	ASSERT_NE(buf[2], TRANS_PIXEL);
	ASSERT_EQ(buf[3], TRANS_PIXEL);
	ASSERT_NE(buf[4], TRANS_PIXEL);
	ASSERT_EQ(buf[5], TRANS_PIXEL);
	ASSERT_NE(buf[6], TRANS_PIXEL);
	ASSERT_EQ(buf[7], TRANS_PIXEL);
}

static Img_c getCrossImage()
{
	Img_c image(3, 3);
	img_pixel_t *buf = image.wbuf();
	buf[1] = buf[3] = buf[4] = buf[5] = buf[7] = 200;	// +
	EXPECT_TRUE(image.has_transparent());

	return image;
}

TEST(ImageBasic, Spectrify)
{
	ConfigData config = {};

	config.miscInfo.invis_colors[0] = 2;
	config.miscInfo.invis_colors[1] = 10;
	
	Img_c image = getCrossImage();

	Img_c specter = image.spectrify(config);
	ASSERT_FALSE(specter.is_null());
	ASSERT_TRUE(specter.has_transparent());
	ASSERT_EQ(specter.width(), 3);
	ASSERT_EQ(specter.height(), 3);
	const img_pixel_t *sbuf = specter.buf();
	ASSERT_EQ(sbuf[0], TRANS_PIXEL);
	ASSERT_GE(sbuf[1], 2);
	ASSERT_LE(sbuf[1], 10);
	ASSERT_EQ(sbuf[2], TRANS_PIXEL);
	ASSERT_GE(sbuf[3], 2);
	ASSERT_LE(sbuf[3], 10);
	ASSERT_GE(sbuf[4], 2);
	ASSERT_LE(sbuf[4], 10);
	ASSERT_GE(sbuf[5], 2);
	ASSERT_LE(sbuf[5], 10);
	ASSERT_EQ(sbuf[6], TRANS_PIXEL);
	ASSERT_GE(sbuf[7], 2);
	ASSERT_LE(sbuf[7], 10);
	ASSERT_EQ(sbuf[8], TRANS_PIXEL);
}

TEST(ImageBasic, SpectrifySingleColor)
{
	ConfigData config = {};

	config.miscInfo.invis_colors[0] = 2;
	config.miscInfo.invis_colors[1] = 0;

	Img_c image = getCrossImage();

	Img_c specter = image.spectrify(config);
	ASSERT_FALSE(specter.is_null());
	ASSERT_TRUE(specter.has_transparent());
	ASSERT_EQ(specter.width(), 3);
	ASSERT_EQ(specter.height(), 3);
	const img_pixel_t *sbuf = specter.buf();
	ASSERT_EQ(sbuf[0], TRANS_PIXEL);
	ASSERT_EQ(sbuf[1], 2);
	ASSERT_EQ(sbuf[2], TRANS_PIXEL);
	ASSERT_EQ(sbuf[3], 2);
	ASSERT_EQ(sbuf[4], 2);
	ASSERT_EQ(sbuf[5], 2);
	ASSERT_EQ(sbuf[6], TRANS_PIXEL);
	ASSERT_EQ(sbuf[7], 2);
	ASSERT_EQ(sbuf[8], TRANS_PIXEL);
}

//==================================================================================================

//
// Fixture with a palette
//
class ImageFixture : public ::testing::Test
{
protected:
	void SetUp() override;

	Palette palette;
};

//
// Setup the palette
//
void ImageFixture::SetUp()
{
	makeCommonPalette(palette);
}

TEST_F(ImageFixture, CreateLightSprite)
{
	Img_c image = Img_c::createLightSprite(palette);
	ASSERT_EQ(image.width(), 11);
	ASSERT_EQ(image.height(), 11);

	static const img_pixel_t expected[11 * 11] = {
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0x2d, 0x32, 0x32, 0x2d, 0xff, 0xff, 0xff,
		0xff, 0xff, 0x2d, 0x5a, 0x88, 0x8d, 0x8d, 0x88, 0x5a, 0x2d, 0xff,
		0xff, 0xff, 0x5a, 0x8d, 0xe7, 0xed, 0xed, 0xe7, 0x8d, 0x5a, 0xff,
		0xff, 0x2d, 0x88, 0xe7, 0xed, 0xee, 0xee, 0xed, 0xe7, 0x88, 0x2d,
		0xff, 0x32, 0x8d, 0xed, 0xee, 0xef, 0xef, 0xee, 0xed, 0x8d, 0x32,
		0xff, 0x32, 0x8d, 0xed, 0xee, 0xef, 0xef, 0xee, 0xed, 0x8d, 0x32,
		0xff, 0x2d, 0x88, 0xe7, 0xed, 0xee, 0xee, 0xed, 0xe7, 0x88, 0x2d,
		0xff, 0xff, 0x5a, 0x8d, 0xe7, 0xed, 0xed, 0xe7, 0x8d, 0x5a, 0xff,
		0xff, 0xff, 0x2d, 0x5a, 0x88, 0x8d, 0x8d, 0x88, 0x5a, 0x2d, 0xff,
		0xff, 0xff, 0xff, 0xff, 0x2d, 0x32, 0x32, 0x2d, 0xff, 0xff, 0xff,
	};

	ASSERT_EQ(memcmp(image.buf(), expected, sizeof(expected)), 0);
}

TEST_F(ImageFixture, CreateMapSpotSprite)
{
	struct Case
	{
		int r, g, b;
		const img_pixel_t data[32 * 32];
	};
	static const Case cases[] = {
		{255, 0,   0, {
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff,
			0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff,
			0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff,
			0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0xc8, 0xc8, 0xc8, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0xc8, 0xc8, 0xc8, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0xc8, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0xc8, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0xa0, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0xa0, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x78, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x78, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x50, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x28, 0x50, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		}},
		{0, 255,   0, {
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x1e, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x1e, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff,
			0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x1e, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x1e, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff,
			0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x1e, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x1e, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff,
			0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x1e, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x1e, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x1e, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x1e, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x1e, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x1e, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x1e, 0x23, 0x23, 0x23, 0x23, 0x23, 0x1e, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x1e, 0x23, 0x23, 0x23, 0x23, 0x23, 0x1e, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x1e, 0x23, 0x23, 0x23, 0x1e, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x1e, 0x23, 0x23, 0x23, 0x1e, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x1e, 0x23, 0x1e, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x1e, 0x23, 0x1e, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x1e, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x1e, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x1e, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x19, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x19, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x14, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x14, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0f, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0a, 0x0f, 0x0a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		}},
		{0,   0, 255, {
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff,
			0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff,
			0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff,
			0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x04, 0x04, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x04, 0x04, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x02, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x02, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x01, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		}},
	};
	for(const Case &caze : cases)
	{
		Img_c image = Img_c::createMapSpotSprite(palette, caze.r, caze.g, caze.b);
		ASSERT_EQ(image.width(), 32);
		ASSERT_EQ(image.height(), 32);

		ASSERT_EQ(memcmp(image.buf(), caze.data, sizeof(caze.data)), 0);
	}
}

TEST_F(ImageFixture, CreateDogSprite)
{
	Img_c image = Img_c::createDogSprite(palette);
	ASSERT_EQ(image.width(), 44);
	ASSERT_EQ(image.height(), 26);

	static const img_pixel_t expected[44 * 26] = {
		255, 255, 255, 255, 255, 255, 255, 242, 242, 242, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 255, 242, 131, 131, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 242, 131, 131, 131, 131, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
		255, 242, 242, 242, 242, 131, 0, 131, 131, 131, 131, 131, 130, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
		242, 45, 45, 130, 131, 131, 131, 45, 131, 131, 131, 131, 130, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
		242, 45, 85, 130, 130, 130, 131, 131, 130, 131, 131, 130, 131, 131, 242, 242, 242, 242, 242, 242, 242, 242, 242, 242, 242, 242, 242, 242, 242, 242, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 255,
		242, 45, 45, 85, 85, 130, 130, 131, 85, 130, 130, 85, 131, 0, 0, 0, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 242, 242, 255, 255, 255, 255, 255, 255, 255, 255, 242, 131, 242,
		255, 242, 45, 45, 85, 85, 85, 130, 45, 85, 85, 45, 0, 0, 0, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 85, 242, 242, 242, 255, 255, 255, 242, 242, 131, 242, 255,
		255, 255, 242, 45, 45, 45, 242, 242, 45, 85, 45, 0, 0, 0, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 85, 85, 85, 85, 130, 130, 242, 242, 242, 131, 131, 131, 242, 255,
		255, 255, 255, 242, 242, 242, 255, 255, 242, 45, 45, 0, 0, 85, 85, 85, 85, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 85, 85, 130, 130, 130, 130, 131, 131, 131, 131, 131, 242, 242, 255, 255,
		255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 0, 0, 131, 130, 130, 130, 85, 85, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 85, 85, 130, 131, 131, 131, 131, 131, 131, 131, 242, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 242, 131, 131, 130, 131, 130, 130, 85, 45, 45, 45, 45, 45, 45, 45, 45, 45, 85, 85, 130, 130, 131, 242, 242, 242, 242, 242, 242, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 131, 131, 131, 131, 130, 130, 130, 45, 45, 45, 45, 45, 45, 45, 85, 85, 130, 131, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 131, 131, 131, 131, 131, 131, 130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 131, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 45, 45, 131, 131, 131, 131, 131, 131, 131, 131, 242, 242, 242, 242, 131, 131, 131, 131, 131, 131, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 45, 45, 131, 131, 131, 131, 131, 242, 242, 242, 255, 255, 255, 255, 242, 45, 45, 131, 131, 131, 131, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 45, 45, 131, 131, 131, 131, 242, 242, 255, 255, 255, 255, 255, 255, 255, 242, 45, 45, 45, 45, 131, 131, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 85, 45, 131, 131, 131, 242, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 85, 85, 45, 45, 242, 131, 131, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 255, 255, 255, 242, 85, 45, 45, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 85, 85, 85, 85, 242, 242, 131, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 255, 242, 242, 130, 85, 45, 242, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 130, 130, 130, 242, 255, 242, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 242, 131, 131, 130, 130, 242, 242, 131, 131, 131, 242, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 130, 130, 130, 242, 242, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 242, 131, 131, 131, 131, 242, 255, 255, 242, 131, 131, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 130, 131, 131, 242, 242, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 242, 242, 242, 242, 255, 255, 255, 255, 242, 131, 131, 131, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 131, 131, 131, 242, 255, 255, 242, 131, 131, 242, 255, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 242, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 131, 131, 131, 242, 255, 255, 255, 242, 131, 131, 242, 255, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 131, 131, 131, 242, 255, 255, 255, 242, 131, 131, 131, 242, 255, 255, 255, 255, 255, 255,
		255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 242, 242, 242, 255, 255, 255, 255, 255, 242, 242, 242, 255, 255, 255, 255, 255, 255, 255,
	};

	ASSERT_EQ(memcmp(image.buf(), expected, sizeof(expected)), 0);
}
